/*

  Gloss Mapping
  
  Gloss maps alter the specular component of a material on a texel-by-texel
  basis.
  
  To draw a gloss map, you first render the object with the specular color
  reset:
  
      Color specular = materialSpecular (FRONT); // Save the specular color.

      materialSpecular (FRONT, 0, 0, 0); // Reset it to black.
      list.call (); // Draw the object.

  Then you draw it with the diffuse, ambient, and emission colors reset
  using a gloss map texture:

      // Reset the other colors to black.
      materialDiffuse (FRONT, 0, 0, 0);
      materialAmbient (FRONT, 0, 0, 0);
      materialEmission (FRONT, 0, 0, 0);

      // Restore the color we saved.
      materialSpecular (FRONT, specular);

      // If enabled it would ignore the gloss map.
      disable (SEPARATE_SPECULAR_COLOR);

      // If enabled it would cause haloing.
      disable (FOG);

      // Setup blending.
      enable (BLEND);
      blendFunc (ONE, ONE);

      texture.enable (); // Bind the texture.
      list.call (); // Draw the object.

  Note that SEPARATE_SPECULAR_COLOR must be disabled, as enabling it
  makes the specular highlight not depend upon the texture, the exact
  opposite of what we want.  This technique duplicates its purpose in
  any case.

  Gloss maps are generally LUMINANCE textures.  Even simply copying
  your main texture and converting it to grayscale is effective and reduces
  lighting artifacts, allowing you to have a shiny model at a lower level of
  detail.

*/

version = ShowBrush;
version = ShowGL;

import net.BurtonRadons.dig.main;
import net.BurtonRadons.dig.gl;

/** This is a simple library object that draws a HAL 9000 eye-style button. */

version (ShowBrush)
class HalButton
{
public:
    static BrushGradient gradient; /**< The gray gradient to use. */
    static BrushGradient counter; /**< The counter-gradient to use. */
    static BrushGradient fill; /**< The eye fill color, modified by draw. */
    static BrushGradient highlight; /**< The eye highlight. */

    float sradius = 16; /**< The current draw radius. */
    Color scolor; /**< The current eye color. */

    /** Sets the eye to use a blue color. */
    this () { scolor = AColor (128, 128, 255); }

    /** Sets up the gradients. */
    static this ()
    {
        Color a = AColor (0, 0, 0);
        Color b = AColor (196, 196, 196);
        Color hia = AColor (255, 255, 255);
        Color hib = AColor (255, 255, 255, 0);

        with (gradient = new BrushGradient ())
        {
            center (32, 32);
            scale (64 * 1.5);
            opAdd (0, a);
            opAdd (1, b);
            radial (true);
        }

        with (counter = new BrushGradient ())
        {
            center (-64, -64);
            scale (96 * 1.5);
            opAdd (0, a);
            opAdd (1, b);
            radial (true);
        }

        with (fill = new BrushGradient ())
        {
            center (0, -16);
            scale (32);
            opAdd (0, a);
            opAdd (1, b);
        }

        with (highlight = new BrushGradient ())
        {
            center (0, -32 - 4 * 2);
            scale (4 * 4);
            opAdd (0, hia);
            opAdd (1, hib);
        }
    }
    
    /** Set the color of the eye. */
    void color (Color color) { scolor = color; }

    /** Get the color of the eye. */
    Color color () { return scolor; }

    /** Set the drawing radius. */
    void radius (float value) { sradius = value; }

    /** Draw the eye on the bitmap centered at a specific point. */
    void draw (Bitmap bitmap, float x, float y)
    {
        float r = sradius / 64;

        fill.set (0, AColor (scolor.r * 3/4, scolor.g * 3/4, scolor.b * 3/4));
        fill.set (1, scolor);

        gradient.center (x + 32 * r, y + 32 * r);
        gradient.scale (96 * r);

        counter.center (x - 64 * r, y - 64 * r);
        counter.scale (144 * r);

        fill.center (x, y - 16 * r);
        fill.scale (32 * r);

        highlight.center (x, y - 40 * r);
        highlight.scale (16 * r);

        with (bitmap)
        {
            circle (x, y, 64 * r, gradient);
            circle (x, y, 48 * r, counter);
            circle (x, y, 40 * r, fill);
            circle (x, y - 32 * r, 16 * r, 8 * r, highlight);
        }
    }
};

/* This is a small testing program that shows the button, text, menu, and
 * canvas widgets, as well as a little of what Bitmap can do.
 */

class MyFrame : Frame
{
    Button fileSelector, fontSelector;
    Label text; /* A text box that changes according to stuff. */

    version (ShowBrush)
    {
        Button changeColor;
        ColorSelector colorSelector; /* The color selector object.  We keep the object around to share the default colors. */
        HalButton hal; /* The HAL 9000 eye button. */
        Bitmap bitmap; /* The bitmap we draw the button upon. */
        Canvas canvas; /* The canvas we draw the bitmap upon. */
    }

    version (ShowGL)
        CanvasGL frame; /* The frame, the frame! */

    this ()
    {
        int r = 0;

        super ();
        caption ("Testing Nothing");
        onLButtonDown.add (&mouseDown);
        onLButtonUp.add (&mouseUp);
        onMouseMove.add (&mouseMove);
        icon (null);

        version (ShowBrush)
        {
            colorSelector = new ColorSelector ();
            hal = new HalButton ();

            with (changeColor = new Button (this))
            {
                caption ("&Change Color");
                onClick.add (&changeColorClick);
                bordered (true);
                alignRight (true);
                gridAddRow (0, r);
                sticky ("<>^");
            }
        }

        with (fileSelector = new Button (this))
        {
            caption ("&File Selector");
            onClick.add (&fileSelectorClick);
            alignLeft (true);
            gridAddRow (0, r);
            sticky ("<>^");
        }

        with (text = new Label (this))
        {
            caption ("Hello, world!");
            gridAddRow (0, r);
            sticky ("<>^");
        }

        with (fontSelector = new Button (this))
        {
            caption ("F&ont Selector");
            onClick.add (&fontSelectorClick);
            gridAddRow (0, r);
            sticky ("<>^");
        }

        version (ShowBrush)
        {
            with (canvas = new Canvas (this))
            {
                widthAndHeight (128, 128);
                onPaint.add (&paintCanvas);
                this.bitmap = new Bitmap (width (), height ());
                doubleBuffer (false);
                gridAddRow (0, r);
                cursor (Cursor.Working);
            }
        }

        version (ShowGL)
        {
            with (frame = new CanvasGL (this))
            {
                widthAndHeight (128, 128);
                onPaint.add (&paintFrame);
                onLButtonUp.add (&changeMaterial);
                gridAddRow (0, r);
            }
        }

        maximizable (false);
        minimizable (false);
        resizable (false);

        version (ShowGL)
            timer (10, &paintFrame);
    }

    version (ShowGL)
    {
        GLlist list;
        GLtexture texture, glossmap;
        float theta = 0;
        GLlight light;

        struct Material
        {
            char [] name;
            GLfloat [3] a;
            GLfloat [3] d;
            GLfloat [3] s;
            GLfloat shininess;
            GLfloat opacity;

            void setup ()
            {
                with (gl)
                {
                    GLenum face = FRONT;

                    materialDiffuse (face, d [0], d [1], d [2], opacity);
                    materialEmission (face, 0.2, 0.2, 0.2);
                    materialSpecular (face, s [0], s [1], s [2], opacity);
                    materialAmbient (face, a [0], a [1], a [2], opacity);
                    materialShininess (face, shininess);
                }
            }
        }

        Material [] materials =
        [
            { "Gold",             [ 0.247, 0.199, 0.074 ], [ 0.751, 0.606, 0.226 ], [ 0.628, 0.556, 0.366 ], 51.200, 1.00 },
            { "Gold, Polished",   [ 0.247, 0.224, 0.064 ], [ 0.346, 0.314, 0.090 ], [ 0.797, 0.724, 0.208 ], 83.200, 1.00 },
            { "Chrome",           [ 0.250, 0.250, 0.250 ], [ 0.400, 0.400, 0.400 ], [ 0.775, 0.775, 0.775 ], 76.800, 1.00 },
            { "Pewter",           [ 0.105, 0.059, 0.114 ], [ 0.427, 0.470, 0.541 ], [ 0.333, 0.333, 0.522 ],  9.846, 1.00 },
            { "Emerald",          [ 0.022, 0.175, 0.022 ], [ 0.076, 0.614, 0.076 ], [ 0.633, 0.728, 0.633 ], 76.800, 0.55 },
            { "Ruby",             [ 0.175, 0.012, 0.012 ], [ 0.614, 0.041, 0.041 ], [ 0.728, 0.627, 0.627 ], 76.800, 0.55 },
            { "Silver, Polished", [ 0.231, 0.231, 0.231 ], [ 0.278, 0.278, 0.278 ], [ 0.774, 0.774, 0.774 ], 89.600, 1.00 },
            { "Silver",           [ 0.192, 0.192, 0.192 ], [ 0.508, 0.508, 0.508 ], [ 0.508, 0.508, 0.508 ], 51.200, 1.00 },
            { "Brass",            [ 0.329, 0.224, 0.027 ], [ 0.780, 0.569, 0.113 ], [ 0.992, 0.941, 0.808 ], 27.897, 1.00 },
            { "Bronze",           [ 0.213, 0.128, 0.054 ], [ 0.714, 0.428, 0.181 ], [ 0.394, 0.272, 0.167 ], 25.600, 1.00 },
            { "Bronze, Polished", [ 0.250, 0.148, 0.065 ], [ 0.400, 0.237, 0.104 ], [ 0.775, 0.459, 0.201 ], 76.800, 1.00 },
            { "Copper",           [ 0.191, 0.074, 0.023 ], [ 0.704, 0.270, 0.083 ], [ 0.257, 0.138, 0.086 ], 12.800, 1.00 },
            { "Copper, Polished", [ 0.230, 0.088, 0.028 ], [ 0.551, 0.212, 0.066 ], [ 0.581, 0.223, 0.070 ], 51.200, 1.00 },
            { "Jade",             [ 0.135, 0.223, 0.158 ], [ 0.540, 0.890, 0.630 ], [ 0.317, 0.317, 0.317 ], 12.800, 0.95 },
            { "Obsidian",         [ 0.054, 0.050, 0.066 ], [ 0.183, 0.170, 0.225 ], [ 0.333, 0.329, 0.346 ], 38.400, 0.82 },
            { "Pearl",            [ 0.250, 0.207, 0.207 ], [ 1.000, 0.829, 0.829 ], [ 0.297, 0.297, 0.297 ], 11.264, 0.92 },
            { "Turquoise",        [ 0.100, 0.187, 0.175 ], [ 0.396, 0.742, 0.691 ], [ 0.297, 0.308, 0.307 ], 12.800, 0.80 },
            { "Plastic, Black",   [ 0.000, 0.000, 0.000 ], [ 0.010, 0.010, 0.010 ], [ 0.500, 0.500, 0.500 ], 32.000, 1.00 },
            { "Rubber, Black",    [ 0.020, 0.020, 0.020 ], [ 0.010, 0.010, 0.010 ], [ 0.400, 0.400, 0.400 ], 10.000, 1.00 },
        ];

        int currentMaterial = 0;

        class MaterialSelect
        {
            MyFrame frame;
            int value;

            this (MyFrame frame, int value)
            {
                this.frame = frame;
                this.value = value;
            }

            void run (Event e)
            {
                frame.currentMaterial = value;
            }
        }

        void changeMaterial (Event e)
        {
            Menu menu;

            with (menu = new Menu (this))
            {
                for (int c; c < materials.length; c ++)
                {
                    char [] name = materials [c].name;

                    if (c == currentMaterial)
                        name ~= " (current)";
                    opAdd (name, &(new MaterialSelect (this, c)).run);
                }
                exec ();
            }
        }

        void drawGlossmap (GLlist list,
                           GLtexture texture,
                           GLtexture glossmap)
        {
            with (gl)
            {
                GLenum face = FRONT;
                Color diffuse = materialDiffuse (face);
                Color ambient = materialAmbient (face);
                Color emission = materialEmission (face);
                Color specular = materialSpecular (face);
                bit fog, blend;

            /* Setup the material parameters - clear specular for the gloss map. */
                materialSpecular (face, 0, 0, 0);
                texture.enable ();
                list.call ();

            /* Now draw the gloss map. */
                materialSpecular (face, specular);
                materialDiffuse (face, 0, 0, 0);
                materialAmbient (face, 0, 0, 0);
                materialEmission (face, 0, 0, 0);
                fog = disableAndGet (FOG);
                blend = enableAndGet (BLEND);
                blendFunc (SRC_ALPHA, ONE);
                scale (0.5);

                glossmap.enable ();
                list.call ();

            /* Reset our state changes */
                materialDiffuse (face, diffuse);
                materialAmbient (face, ambient);
                materialEmission (face, emission);
                store (BLEND, blend);
                store (FOG, fog);
            }
        }

        void paintFrame (Event e)
        {
            Color ambient, diffuse, specular;
            GLfloat shininess;

            Material *m = &materials [currentMaterial];

            with (gl)
            {
                if (!texture)
                {
                    const int w = 128 * 2;
                    const int h = 128 * 2;
                    GLubyte [] data;

                    data = new GLubyte [w * h * 3];

                    /* Another use for brushes. */
                    BrushColor cbrush = new BrushColor (Color.White);//diffuse);
                    BrushNoise nbrush;
                    
                    with (nbrush = new BrushNoise (cbrush))
                    {
                        scale (32 * 4);
                        uniform (true);
                    }

                    for (int x; x < w; x ++)
                        for (int y; y < h; y ++)
                        {
                            Color c = nbrush.eval (x, y);

                            data [(x + y * w) * 3 + 0] = c.r;
                            data [(x + y * w) * 3 + 1] = c.g;
                            data [(x + y * w) * 3 + 2] = c.b;
                        }

                    with (texture = gl.textureNew ())
                    {
                        image (w, h, RGB, UNSIGNED_BYTE, data);
                        minFilter (LINEAR_MIPMAP_LINEAR);
                    }

                    cbrush = new BrushColor (AColor (128, 128, 128));
                    nbrush = new BrushNoise (cbrush);
                    nbrush.scale (256);
                    nbrush.uniform (true);
                    nbrush.offset (640, 480);

                    for (int x; x < w; x ++)
                        for (int y; y < h; y ++)
                        {
                            Color c = nbrush.eval (x, y);

                            data [(x + y * w) * 3 + 0] = c.r;
                            data [(x + y * w) * 3 + 1] = c.g;
                            data [(x + y * w) * 3 + 2] = c.b;
                        }

                    with (glossmap = textureNew ())
                        image (w, h, RGB, UNSIGNED_BYTE, data);

                    delete data;
                }

                frame.beginPaint ();

                clearColor (backgroundColor ());
                clear (COLOR_BUFFER_BIT | DEPTH_BUFFER_BIT);

            /* Setup the projection and modelview matrices. */
                setupPerspective (60, 1, 0.1, 400);

                matrixMode (MODELVIEW);
                translate (0, 0, -3);
                rotate (40, 1, 0, 0);
                rotate (theta, 0, 1, 0);

            /* Setup the light. */
                enable (LIGHTING);
                with (light = getLight (LIGHT0))
                {
                    enable ();
                    position (0.2, 0.7, 1);
                    diffuse (1, 1, 1);
                    ambient (0.5, 0.5, 0.5);
                    specular (1, 1, 1);
                    attenuation (1, 0, 0);
                    delete light;
                }

            /* Store miscellaneous state. */
                enable (LIGHT_MODEL_LOCAL_VIEWER);
                //enable (SEPARATE_SPECULAR_COLOR); // cancels out gloss mapping.
                disable (DITHER);

            /* Setup the depth buffer. */
                //enable (DEPTH_TEST);
                depthFunc (LEQUAL);

            /* Setup fogging; we fog just a slice to fake anti-aliasing. */
                //enable (FOG);
                fogDistanceMode (EYE_RADIAL);
                fogMode (LINEAR);
                fogStart (1.9);
                fogEnd (fogStart () + 6.1);
                fogColor (backgroundColor ());

            /* Setup the texturing matrix. */
                matrixMode (TEXTURE);
                loadIdentity ();
                scale (4.0);

            /* Prepare the drawing list. */
                if (list === null)
                {
                    with (list = listNew ())
                    {
                        compile ();
                        sphere (1.5, 128, 64);
                        compileEnd ();
                    }
                }

            /* Setup culling. */
                enable (CULL_FACE);
                cullFace (BACK);

            /* Draw the object */
                m.setup ();
                if (m.opacity == 1)
                {
                    disable (BLEND);
                    drawGlossmap (list, texture, glossmap);
                }
                else
                {
                    enable (BLEND);
                    blendFunc (SRC_ALPHA, ONE_MINUS_SRC_ALPHA);
                    enable (CULL_FACE);
                    cullFace (FRONT);
                    drawGlossmap (list, texture, glossmap);

                    blendFunc (SRC_ALPHA, ONE_MINUS_SRC_ALPHA);
                    cullFace (BACK);
                    drawGlossmap (list, texture, glossmap);
                }


            /* Finished */
                frame.endPaint ();
            }
            
            theta = theta + 0.5;
            timer (10, &paintFrame);
        }
    }

    version (ShowBrush)
    void paintCanvas (Event e)
    {
        canvas.beginPaint ();

        with (bitmap)
        {
            int w = width () / 2;
            int h = height () / 2;

            clear (backgroundColor ());
            hal.radius ((w + h) / 2);
            hal.draw (this.bitmap, w, h);
            blit (canvas, 0, 0);
        }

        canvas.endPaint ();
    }

    version (ShowBrush)
    void colorSelectorRun (Event e)
    {
        with (colorSelector)
        {
            color = hal.color ();
            if (run ())
            {
                hal.color (color);
                canvas.paint ();
            }
        }
    }

    version (ShowBrush)
    void changeColorClick (Event e)
    {
        Menu menu, popupMenu;

        with (popupMenu = new Menu (this))
        {
            opAdd ("Bleck", &MakeHalPuke);
        }

        with (menu = new Menu (this))
        {
            opAdd ("bop", "Disabled");
            disabled ("bop", true);
            separator ();
            opAdd ("New hotness", &MakeHalRed);
            highlight (true);
            opAdd ("Old and busted", &MakeHalGreen);
            checked (true);
            opAdd ("Color Selector", &colorSelectorRun);
            popup ("Popup", popupMenu);
            exec ();
        }
    }

    version (ShowBrush)
    {
        void MakeHalRed (Event e) { hal.color (AColor (255, 0, 0)); canvas.paint (); }
        void MakeHalGreen (Event e) { hal.color (AColor (64, 196, 128)); canvas.paint (); }
        void MakeHalPuke (Event e) { hal.color (AColor (32, 96, 0)); canvas.paint (); }
    }
    
    void fontSelectorClick (Event e)
    {
        with (new FontSelector (text.font ()))
        {
            Font font = run ();

            if (font)
            {
                version (ShowBrush)
                    changeColor.font (font);
                fileSelector.font (font);
                fontSelector.font (font);
                text.font (font);
            }
        }
    }

    void fileSelectorClick (Event e)
    {
        FileSelector sel = new FileSelector (this, false);
        char [] [] result;

        sel.title = "Select a file!";
        sel.addFilter ("All Files", "*");
        sel.addFilter ("Text Files", "*.txt;*.bak;README");
        result = sel.run ();

        if (!result.length)
            printf ("(canceled)\n");
        else if (result.length == 1)
            printf ("%.*s\n", result [0]);
        else
        {
            printf ("multiple return:\n");
            for (int c; c < result.length; c ++)
                printf ("    %.*s\n", result [c]);
        }

        for (int c; c < result.length; c ++)
            delete result [c];
        delete result;

        //messageBox ("Not so fast!");
    }

    void mouseDown (Event e) { text.caption (fmt ("Left Down %dx%d", e.x, e.y)); }
    void mouseUp (Event e) { text.caption (fmt ("Left Up %dx%d", e.x, e.y)); }

    int mx, my;

    void mouseMove (Event e)
    {
        if (mx == e.x && my == e.y)
            return;
        text.caption (fmt ("%dx%d", e.x, e.y));
        mx = e.x;
        my = e.y;
    }
}

void main ()
{
    (new MyFrame ()).showModal ();
}
